<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use Laravel\Socialite\Facades\Socialite;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Auth;
use App\Models\UserOauth;
use App\Models\ErrorLog;
use App\Models\Settings;
use App\Facades\Theme;

class OauthController extends Controller
{
    public $service;

    public function __construct()
    {
        $this->service = request()->route('service');

        $services = ['github', 'discord', 'google'];
        if(!in_array($this->service, $services)) {
            return redirect()->back()->send();
        }

        $this->setConfig();
    }

    public function connect()
    {
        try {
            return Socialite::driver($this->service)->redirect();
        } catch(\Exception $error) {
            ErrorLog::catch('oauth::connect::'. $this->service, $error->getMessage());
            return redirect()->route('user.settings')->withError("Authentication Failed: Please contact an Administrator, errors have been logged.");
        }
    }

    public function remove()
    {        
        if(Auth::user()->oauthService($this->service)->exists()) {
            Auth::user()->oauthService($this->service)->first()->delete();
        }

        return redirect()->route('user.settings')->withSuccess($this->service. ' was disconnected from your account.');
    }

    public function callback()
    {
        try {
            $user = Socialite::driver($this->service)->user();

            // ensure whether the driver/email combination is unique
            if(UserOauth::where(['driver' => $this->service, 'email' => $user->getEmail()])->exists()) {
                return redirect()->route('user.settings')->withError($this->service. ' is already connected to your account.');
            }

            return $this->{$this->service}($user);
        } catch(\Exception $error) {
            ErrorLog::catch('oauth::callback::'. $this->service, $error->getMessage());
            return redirect()->route('user.settings')->withError("Something went wrong, please try again.");
        }
    }

    // store the user for google
    protected function google($user)
    {
        // ensure user is verified on Google
        if(!$user->user['verified_email']) {
            return redirect()->route('user.settings')->withError('Your Google account is not verified.');
        }

        UserOauth::UpdateOrCreate([
            'user_id' => Auth::user()->id,
            'driver' => $this->service,
            'email' => $user->getEmail(),
            'data' => $user->user,
        ]);

        return redirect()->route('user.settings')->withSuccess('Successfully connected '. $this->service . ' to your account.');
    }

    // store the user for github
    protected function github($user)
    {
        UserOauth::UpdateOrCreate([
            'user_id' => Auth::user()->id,
            'driver' => $this->service,
            'email' => $user->getEmail(),
            'external_profile' => $user->user['html_url'],
            'data' => $user->user,
        ]);

        return redirect()->route('user.settings')->withSuccess('Successfully connected '. $this->service . ' to your account.');
    }

    // store the user for github
    protected function discord($user)
    {
        // ensure user is verified on discord
        if(!$user->user['verified']) {
            return redirect()->route('user.settings')->withError('Your Discord account is not verified.');
        }

        UserOauth::UpdateOrCreate([
            'user_id' => Auth::user()->id,
            'driver' => $this->service,
            'email' => $user->getEmail(),
            'data' => $user->user,
        ]);

        return redirect()->route('user.settings')->withSuccess('Successfully connected '. $this->service . ' to your account.');
    }

    /**
     * This function dynamically sets the values for config/services.php 
     * so that oauth services can retrieve settings set by the user in the admin area
     *
     * @return void
     */
    protected function setConfig()
    {
        config([
            'services.'.$this->service.'.client_id' => Settings::getJson('encrypted::oauth::'. $this->service, 'client_id')
        ]);

        config([
            'services.'.$this->service.'.client_secret' => Settings::getJson('encrypted::oauth::'. $this->service, 'client_secret')
        ]);

        config([
            'services.'.$this->service.'.redirect' => config('app.url') . '/oauth/'.$this->service.'/redirect'
        ]);
    }

}
