<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use App\Facades\AdminTheme as Theme;
use App\Models\User;
use App\Models\Admin\Group;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\UserIp;
use App\Models\Device;
use App\Models\Punishment;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{

    public function index()
    {
        $page = $_GET['page'] ?? 1;

        if(array_key_exists('sort', $_GET) and $_GET['sort'] == 'online'){
            // Online
            $users = User::getOnlineUsers()->paginate(30, ['*'], 'page', $page);
            $users->appends(['sort' => 'online']);
        } else {
            // All
            $users = User::query()->latest()->paginate(30, ['*'], 'page', $page);
        }
        return Theme::view('users.index', compact('users'));
    }

    public function edit(User $user)
    {
        $groups = Group::query()->get();
        return Theme::view('users.edit', compact('user', 'groups'));
    }

    public function update(Request $request, User $user)
    {
        $request->validate([
            'first_name' => 'required|max:255',
            'last_name' => 'required|max:255',
            'username' => 'required|unique:users,username,' . $user->id . '|max:191|regex:/^[A-Za-z0-9_]+$/',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'groups' => 'nullable|array',
            'password' => 'nullable|min:4|max:255',
            'company_name' => 'max:255',
            'address' => 'max:255',
            'address_2' => 'max:255',
            'country' => 'max:255',
            'city' => 'max:255',
            'region' => 'max:255',
            'zip_code' => 'max:255',
        ]);

        // reset the password if it is set
        if ($request->input('password', false)) {
            $user->password = Hash::make($request->input('password'));
        }

        $user->groups()->sync($request->input('groups'), true);
        $user->first_name = $request->input('first_name');
        $user->last_name = $request->input('last_name');
        $user->email = $request->input('email');
        $user->status = $request->input('status');
        $user->username = $request->input('username');
        $user->is_subscribed = isset($request->is_subscribed) && $request->is_subscribed == "on";
        $user->save();

        $address = $user->address;
        $address->company_name = $request->input('company_name');
        $address->address = $request->input('address');
        $address->address_2 = $request->input('address_2');
        $address->country = $request->input('country');
        $address->city = $request->input('city');
        $address->region = $request->input('region');
        $address->zip_code = $request->input('zip_code');
        $address->save();

        return redirect()->back()->with('success',
            trans('responses.user_update_success',
                ['default' => 'User updated successfully'])
        );
    }

    public function uploadProfilePicture(Request $request, User $user)
    {
        $request->validate([
            'avatar' => 'required|image|max:2048', // Maximum file size is 2MB
        ]);

        // delete avatar if user already has one
        if ($user->avatar !== NULL) {
            Storage::delete('public/avatars/' . $user->avatar);
        }

        if ($request->hasFile('avatar')) {
            $file = $request->file('avatar');
            $filename = Str::random(32) . '.' . $file->getClientOriginalExtension();
            $file->storeAs('public/avatars', $filename);
            $user->avatar = $filename;
            $user->save();
        }

        return redirect()->back()->with('success',
            trans('responses.user_upload_picture_success',
                ['default' => 'Profile picture uploaded successfully'])
        );
    }

    public function resetProfilePicture(User $user)
    {
        // delete avatar if user already has one
        if ($user->avatar !== NULL) {
            Storage::delete('public/avatars/' . $user->avatar);
            $user->avatar = NULL;
            $user->save();
        }

        return redirect()->back()->with('success',
            trans('responses.user_picture_reset',
                ['default' => 'Profile picture has been reset'])
        );
    }

    public function destroy(User $user)
    {
        if($user->is_admin()) {
            return redirect()->back()->withError('This account has administrive privelages. The admin permissions must be removed before you can have your account deleted.');
        }

        if($user->orders()->where('status', 'active')->exists()) {
            return redirect()->back()->withError(
                'This user has active orders. You must terminate the orders before deleting the user'
            );
        }

        if($user->punishments()->exists()) {
            return redirect()->back()->withError(
                'This user has a recorded punishments. You must delete the punishments before deleting the user'
            );
        }

        $user->terminate();

        return redirect('/admin/users')->withSuccess('User has been deleted permanently.');
    }

    public function revokeDevice(User $user, Device $device)
    {
        $device->revoke();
        return redirect()->back()->with('success',
        trans('responses.user_device_updated',
            ['default' => 'Device has been updated'])
        );
    }

    public function destroyDevice(User $user, Device $device)
    {
        $device->delete();
        return redirect()->back()->with('success',
        trans('responses.user_device_deleted',
            ['default' => 'Device has been deleted'])
        );
    }

    public function orders(User $user)
    {
        return Theme::view('users.orders', compact('user'));
    }

    public function invoices(User $user)
    {
        return Theme::view('users.invoices', compact('user'));
    }

    public function emails(User $user)
    {
        return Theme::view('users.emails', compact('user'));
    }

    public function tickets(User $user)
    {
        return Theme::view('users.tickets', compact('user'));
    }

    public function punishments(User $user)
    {
        $punishments = $user->punishments()->latest()->paginate(10);
        return Theme::view('users.punishments', compact('user', 'punishments'));
    }

    public function createPunishment(Request $request, User $user)
    {
        $request->validate([
            'type' => 'required',
            'reason' => 'nullable',
            'expires_at' => 'nullable',
            'ip_address' => 'nullable',
        ]);

        if(request()->getClientIp() == $request->input('ip_address') AND $request->input('type') == 'ipban') {
            return redirect()->back()->with('error', 'So, you want to IP ban your own IP address?');
        }

        if($request->input('type') == 'ipban') {
            return redirect()->back()->with('error', 'Sorry, IP bans have been disabled temporarily');
        }

        if ($user->is_admin()) {
            return redirect()->back()->with('error', 'You cannot punish admin users');
        }

        $user->punish([
            'type' => $request->input('type'),
            'reason' => $request->input('reason'),
            'expiry_date' => $request->input('expires_at'),
            'ip_address' => $request->input('ip_address'),
        ]);

        return redirect()->back()->with('success', 'User has been punished');
    }

    public function activity(User $user)
    {
        $ips = UserIp::query()->where('user_id', $user->id)->orderBy('uses', 'desc');
        return Theme::view('users.activity', compact('user', 'ips'));
    }

    public function search()
    {
        $query = request()->query('query');
        return User::query()->where('username', 'LIKE', "%$query%")->orWhere('email', 'LIKE', "%$query%")->take(5)->get();
    }

    public function activate(User $user)
    {
        $user->status = 'active';
        $user->save();

        return redirect()->back()->with('success',
            trans('responses.user_activate_success',
                ['default' => 'User has been activated.'])
        );
    }

    public function verify(User $user)
    {
        $user->markEmailAsVerified();
        return redirect()->back()->with('success',
            trans('responses.user_verify_success',
                ['default' => 'Email has been verified'])
        );
    }

    public function updateBalance(Request $request, User $user)
    {
        $request->validate([
            'amount' => 'required|numeric',
            'type' => 'required|max:1',
        ]);

        $description = $request->input('description') ?? __('admin.balance_update_by', ['auth_user_username' => auth()->user()->username]);
        
        $user->balance($description, $request->input('type'), $request->input('amount'));
        return redirect()->back()->with('success',
            trans('responses.user_update_balance_success',
                ['default' => 'User balance was updated successfully'])
        );
    }

    public function emailPasswordReset(User $user)
    {
        $user->sendPasswordResetEmail();
        return redirect()->back()->with('success',
            trans('responses.user_email_password_reset',
                ['default' => 'Email with password reset link has been emailed'])
        );
    }

    public function impersonate(Request $request, User $user)
    {
        if($user->is_admin()) {
            return redirect()->back()->withError('You cannot login as admin users.');
        }

        // Store the current user's ID
        $request->session()->put('impersonate', $user->id);
        return redirect('/dashboard');
    }

    public function stopImpersonate(Request $request)
    {
        if(!$request->session()->has('impersonate')) {
            return abort('403');
        }

        $id = session('impersonate');
        $request->session()->forget('impersonate');

        // Redirect somewhere
        return redirect()->route('users.edit', $id);
    }
}
