<?php

namespace App\Facades;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Blade;
use App\Models\Settings;


class Theme
{
    # You may use this function to return active theme view from Controller
    public static function view($path, array $data = []): Factory|View|Application
    {
        return view('client.' . Theme::active()->folder . '.' . $path, $data);
    }

    # You may use this function to return details about current active theme
    public static function active()
    {
        if (!Settings::has('theme')) {
            Settings::put('theme', 'Tailwind');
        }

        return Theme::get(Settings::get('theme'));
    }

    # You may use this function to activate a given theme, if theme cannot be found
    # system will opt for the Default theme
    public static function activate($theme): bool
    {
        if (Theme::get($theme)->name !== $theme) {
            return false;
        }

        if (!Settings::has('theme')) {
            Settings::put('theme', 'Tailwind');
        }

        Settings::put('theme', $theme);
        return true;
    }

    # You may use this function to retrieve information about a specific theme,
    # if theme does not exist it will display the default theme
    public static function get($theme)
    {
        $themes = Theme::list();

        if (Arr::has($themes, $theme)) {
            return $themes[$theme];
        }

        return $themes['Tailwind'];
    }

    # You may use this function to retrieve view path to a specific theme,
    # additionally you may pass optional variable $path to direct to a specific folder
    public static function getViewPath($theme, $path = null): string
    {
        return 'client.' . Theme::get($theme)->folder . '.' . $path;
    }

    public static function path($path): string
    {
        return 'client.' . self::active()->folder . '.' . $path;
    }

    public static function base_path($path = null): string
    {
        $path = $path ? "/$path" : $path;
        return "resources/themes/client/" . Theme::active()->folder . $path;
    }

    public static function serviceView(string $service, string $path): string
    {
        $vendorPath = self::path("vendor.Services." . ucfirst($service) . "." . $path);
        if (\View::exists($vendorPath)) {
            return $vendorPath;
        }
        $themePath = strtolower($service) . '::' . self::path($path);
        if (\View::exists($themePath)) {
            return $themePath;
        }
        return strtolower($service) . '::client.default.' . $path;
    }

    public static function moduleView(string $module, string $path): string
    {
        $vendorPath = self::path("vendor.Modules." . ucfirst($module) . "." . $path);
        if (\View::exists($vendorPath)) {
            return $vendorPath;
        }
        $themePath = strtolower($module) . '::' . self::path($path);
        if (\View::exists($themePath)) {
            return $themePath;
        }
        return strtolower($module) . '::client.default.' . $path;
    }



    public static function assets($path): string
    {
        $theme_assets = '/assets/themes/' . self::active()->folder . '/assets/' . $path;
        if (file_exists(base_path('/public/' .  $theme_assets))) {
            return $theme_assets;
        }
        return self::active()->assets . '/assets/' . $path;
    }

    public static function asset($path): string
    {
        return route('client-assets', ['template' => Theme::active()->folder, 'path' => $path]);
    }

    # Module developers may use this function to return the wrapper of the active theme
    # to use it, add @extends(Theme::wrapper())
    public static function wrapper(): string
    {
        return 'client.' . Theme::active()->folder . '.' . Theme::active()->wrapper;
    }

    # You may use this function to retrieve all available themes
    public static function list(): array
    {
        $dir = base_path('resources/themes/client');
        $themes = [];
        $contents = scandir($dir);

        foreach ($contents as $item) {
            if (is_dir($dir . '/' . $item) && $item != '.' && $item != '..') {
                if (file_exists($dir . '/' . $item . '/theme.php')) {
                    $theme = include $dir . '/' . $item . '/theme.php';
                    if (isset($theme['name']) and $theme['name'] !== NULL) {
                        $themes[$theme['name']] = (object) $theme;
                    }
                }
            }
        }

        return $themes;
    }

    public static function registerComponents(): void
    {
        $activeThemeFolder = self::active()->folder;
        $componentsPath = base_path("resources/themes/client/{$activeThemeFolder}/components");
        Blade::anonymousComponentPath($componentsPath);
    }

    public static function pagination(): string
    {
        return Theme::path('layouts.pagination');
    }
}
