<?php

namespace App\Console\Commands\Cronjobs\Orders;

use Symfony\Component\Console\Helper\ProgressBar;
use Illuminate\Console\Command;
use App\Models\ErrorLog;
use App\Models\Order;
use Carbon\Carbon;

class TerminateSuspendedOrders extends Command
{
    protected $signature = 'cron:orders:terminate-suspended';
    protected $description = 'Terminate suspended orders';

    public function handle()
    {
        $terminated_orders = Order::whereStatus('suspended')->whereNotNull('due_date')->where('due_date', '<', Carbon::now()->subDays(settings('orders::terminate_suspended_after', 7)))->get();

        $this->info("Loaded a list of expired orders ". $terminated_orders->count());
        $progressBar = $this->output->createProgressBar(count($terminated_orders));
        $progressBar->start();

        foreach ($terminated_orders as $order) {

            // make sure one time orders are skipped
            if(!$order->isRecurring()) {
                continue;
            }

            try {
                // email
                $amount_rounded = currency('symbol') . number_format($order->price['renewal_price'], 2);
                app()->setLocale($order->user->language);
                $order->user->email([
                    'subject' => __('admin.email_terminated_subject'),
                    'content' =>
                    settings('email::terminated', "We regret to inform you that your service has been terminated due to overdue payment. Termination can happen due to a couple of reasons. You were late on payment, or you cancelled the service. All data / files / licenses belonging to this service have been deleted or revoked. <br> We hope to see you as a client in the future.") .
                    __('admin.email_terminated_content', [
                        'order_id' => $order->id,
                        'due_date' => $order->due_date->translatedFormat('d M Y'),
                        'period' => $order->period(),
                        'amount' => $amount_rounded,
                        'order_name' => $order->name
                    ]),
                ]);

                $order->terminate();

            } catch(\Exception $error) {
                // Catch any exceptions thrown by the service handler
                // Handle the error appropriately and register it as an event
                ErrorLog::updateOrCreate([
                    'source' => 'cron:orders:suspend-expired',
                    'severity' => 'CRITICAL',
                    'message' => "Automatic termination service was unable to terminate order $order->id - Error: ". $error->getMessage(),
                ]);
            }

            $progressBar->advance();
        }

        $progressBar->finish();
        $this->line(''); // Move to the next line after progress bar completion

        $this->info('Task Completed: all suspended orders above the limit were terminated');

    }
}
